# Simple script to do regression testing. Takes one parameter, the name
# of a file containing a list of tests. Lines beginning with '#' are
# comments and ignored. The script also expects a file named 'the_hosts'
# which contain a list of hosts to run the test on. The only requirement
# is that the first column be a name that you can ssh into. Again, '#'
# designates a comment. Tests are expected to print either '[PASS]' or
# '[FAIL]' at the end. Failing tests get noted in regression.log.

# If a test has 1_ in it's name, it is only run on the first node.
# Similarly, if a test has 2_ in it's name, it is only run on the first
# two nodes.

# example tests file
# foo_test
# bar_test
# # comment
# subdir/baz_test

# example the_hosts file
# foo.bar.com
# 10.0.0.42
# foo

use Cwd;
$dir = cwd();

$argc = @ARGV;
if($argc != 1) {
    print("harness <tests_file>\n");
    exit(1);
}
$tests_file = $ARGV[0];

$test_count=0;
open(TESTS_FILE, $tests_file) or die "Can't open $tests_file\n";
LINE: while ($line = <TESTS_FILE>) {
    next LINE if $line =~ /^#/; # skip comment lines
    next LINE if $line =~ /^\s*$/; # and blank lines
    chop($line);
    push(@tests_list, $line);
    $test_count = $test_count + 1;
}
close(TESTS_FILE);

$host_count=0;
open(HOSTS_FILE, "the_hosts") or die "Can't open hosts file\n";
LINE: while ($line = <HOSTS_FILE>) {
    next LINE if $line =~ /^#/; # skip comment lines
    next LINE if $line =~ /^\s*$/; # and blank lines
    chop($line);
    @items = split(" ", $line);
    push(@hosts_list, $items[0]);
    push(@mx_hosts_list, $items[1]);
    push(@eids_list, $items[2]);
    push(@host_handles, $items[1].":".$items[2]);
    $host_count = $host_count + 1;
}
close(HOSTS_FILE);

open(LOG_FILE, ">regression.log") or die "Can't open log file\n";

@cmdline=("ssh");
$total_pass_count=0;
$total_fail_count=0;
foreach $test (@tests_list) {
    $pass_count=0;
    $cmd = $dir . "/" . $test;
    print "-------------\n";
    print "Starting test $test\n";
    print "-------------\n";
    $counter = 0;

# $loop_count is the number of hosts we want to run on. If a test has
# 1_ or 2_ in it's name, only run on that many hosts.
    if ($test =~ /1_/) {
	$loop_count=1;
    }
    elsif ($test =~ /2_/) {
	$loop_count=2;
    }
    else {
	$loop_count=$host_count;
    }

    $loop_index=0;
    foreach $host_handle (@host_handles) {
	if ($loop_index == $loop_count) {
	    last;
	}
	splice(@cmdline, 1, 3, "$hosts_list[$loop_index]", "$cmd", "$counter");
	$counter = $counter + 1;
	print("@cmdline\n");
	open($host_handle, "@cmdline|");
	sleep(1);
	$loop_index = $loop_index + 1;
    }
    $loop_index=0;
    foreach $host_handle (@host_handles) {
	if ($loop_index == $loop_count) {
	    last;
	}
	print "-------------------\n";
	print "Collecting data for $test $mx_hosts_list[$loop_index] $eids_list[$loop_index]\n";
	print "-------------------\n";
	while(<$host_handle>) {
	    print $_;
	    if ($_ =~ /PASS/) {
#		print LOG_FILE "$test $host [PASS]\n";
		$pass_count = $pass_count + 1;
	    }
	    elsif ($_ =~ /FAIL/) {
		print LOG_FILE "$test $host [FAIL]\n";
		$found_fail=1;
	    }
	}
	close($host_handle);
	$loop_index = $loop_index + 1;
    }

# It is safer to count the number of PASS instead of FAIL since failures
# might not print anything out. It does mean that the log might be
# incomplete.
    if ($loop_count == $pass_count) {
	$total_pass_count = $total_pass_count + 1;
    }
    else {
	$total_fail_count = $total_fail_count + 1;
    }
}
close(LOG_FILE);
print "-------\n";
print "Summary\n";
print "$test_count tests\n";
print "$total_pass_count passed\n";
print "$total_fail_count failed\n";
print "Consult regression.log for details\n";
